<?php

namespace App\Http\Controllers;

use App\Models\Afiliasi;
use App\Models\Jamaah;
use App\Models\Keberangkatan;
use App\Http\Requests\StoreKeberangkatanRequest;
use App\Http\Requests\UpdateKeberangkatanRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Exports\KeberangkatanExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;

class KeberangkatanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Keberangkatan::latest();

        if ($request->filled('tanggal_keberangkatan')) {
            $query->where('tanggal_keberangkatan', $request->tanggal_keberangkatan);
        }

        if ($request->filled('status_pembayaran')) {
            $query->where('status_pembayaran', $request->status_pembayaran);
        }

        $data = $query->paginate(5);

        return view('keberangkatan.index', compact('data'));
    }


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('keberangkatan.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Cari afiliasi berdasarkan kode
        $afiliasi = DB::table('afiliasis')->where('kode_afiliasi', $request->kode_afiliasi)->first();
        if (!$afiliasi) {
            return response()->json([
                'status' => false,
                'message' => 'Data afiliasi tidak ada!'
            ]);
        }
        
        // cek jamaah
        $jamaah = DB::table('jamaahs')->where('no_ktp', $request->no_ktp)->first();
        if (!$jamaah) {
            return response()->json([
                'status' => false,
                'message' => 'Data jamaah tidak ada!'
            ]);
        }

        // cek expired
        if ($request->tanggal_keberangkatan > $jamaah->tanggal_expired) {
            return response()->json([
                'status' => false,
                'message' => 'Passport sudah expired tidak ada!'
            ]);
        }


        // Buat kode keberangkatan random 5 karakter (huruf dan angka)
        $kodeKeberangkatan = 'ELQ_' . strtoupper(Str::random(3));

        Keberangkatan::create([
            'jenis_umroh' => $request->jenis_umroh,
            'no_ktp' => $request->no_ktp,
            'id_afiliasi' => $request->kode_afiliasi,
            'tanggal_keberangkatan' => $request->tanggal_keberangkatan,
            'status_pembayaran' => $request->status_pembayaran,
            'tanggal_pembayaran' => $request->tanggal_pembayaran ?? null,
            'kode_keberangkatan' => $kodeKeberangkatan
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Data afiliasi berhasil disimpan!',
            'kode_keberangkatan' => $kodeKeberangkatan
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show($no_ktp)
    {
        // Mengambil data keberangkatan dan jamaah berdasarkan no_ktp
        $data = DB::table('keberangkatans')
            ->select(
                'jamaahs.title', 'jamaahs.nama', 'jamaahs.no_passport', 'jamaahs.tempat_lahir',
                'jamaahs.tanggal_lahir', 'jamaahs.tanggal_iss', 'jamaahs.tanggal_expired',
                'jamaahs.tempat_iss', 'jamaahs.no_ktp', 'jamaahs.alamat', 'jamaahs.rt',
                'jamaahs.rw', 'jamaahs.kelurahan', 'jamaahs.kecamatan', 'jamaahs.kota',
                'jamaahs.provinsi', 'jamaahs.status_perkawinan', 'jamaahs.pekerjaan', 'jamaahs.nama_ayah',
                'keberangkatans.jenis_umroh', 'keberangkatans.kode_keberangkatan', 'keberangkatans.id_afiliasi', 'keberangkatans.tanggal_keberangkatan',
                'keberangkatans.status_pembayaran', 'keberangkatans.tanggal_pembayaran'
            )
            ->leftJoin('jamaahs', 'jamaahs.no_ktp', '=', 'keberangkatans.no_ktp')
            ->where('keberangkatans.no_ktp', $no_ktp)
            ->first();

        // Mengirimkan data ke view
        return view('keberangkatan.show', compact('data'));
    }



    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $keberangkatan = Keberangkatan::findOrFail($id);
        return view('keberangkatan.edit', compact('keberangkatan'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // Ambil data keberangkatan berdasarkan ID
        $keberangkatan = Keberangkatan::findOrFail($id);

        // Ambil semua data dari request
        $data = $request->all();

        // Update data keberangkatan
        $keberangkatan->update($data);

        // Kembalikan response dalam format JSON
        return response()->json([ 
            'status' => true,
            'message' => 'Keberangkatan berhasil diupdate',
            'result' => $keberangkatan
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $keberangkatan = Keberangkatan::findOrFail($id);
        $keberangkatan->delete();

        return redirect()->route('keberangkatan.index')->with('success', 'Keberangkatan berhasil dihapus');
    }

    public function approve($id)
    {
        $keberangkatan = Keberangkatan::findOrFail($id);

        if ($keberangkatan->status_pembayaran === 'Lunas') {
            $keberangkatan->status_pembayaran = 'Belum Lunas';
            $keberangkatan->tanggal_pembayaran = null;
        } else {
            $keberangkatan->status_pembayaran = 'Lunas';
            $keberangkatan->tanggal_pembayaran = now();
        }

        $keberangkatan->save();

        return response()->json([
            'status'  => true,
            'message' => 'Approve berhasil.'
        ]);
    }

    public function search(Request $request)
    {
        $search = $request->search;

        $data = Keberangkatan::where('no_ktp', 'like', "%$search%")
                                    ->orWhere('kode_keberangkatan', 'like', "%$search%")
                                    ->orWhere('no_ktp', 'like', "%$search%")
                                    ->paginate(5);
        
        return view('keberangkatan.index', compact('data'));
    }

    private function afiliasi($id_afiliasi)
    {
        $response = DB::table('afiliasis')->where('id', $id_afiliasi)->first();

        return $response;
    }

    public function export(Request $request) 
    {
        return Excel::download(
            new KeberangkatanExport($request->tanggal_keberangkatan, $request->status_pembayaran),
            'data-keberangkatan.xlsx'
        );

    }



}
