<?php

namespace App\Http\Controllers;

use App\Models\Jamaah;
use App\Models\Keberangkatan;
use App\Http\Requests\StoreJamaahRequest;
use App\Http\Requests\UpdateJamaahRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\JamaahExport;
use App\Http\Controllers\PortalController;

class JamaahController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Jamaah::latest();

        if ($request->filled('kota')) {
            $query->where('kota', $request->kota);
        }
        
        $data = $query->paginate(5);
        return view('jamaah.index', compact('data'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    private function afiliasi($id_afiliasi)
    {
        $response = DB::table('afiliasis')->where('id', $id_afiliasi)->first();

        return $response;
    }
    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Cari afiliasi berdasarkan kode
        $afiliasi = DB::table('afiliasis')->where('kode_afiliasi', $request->id_afiliasi)->first();
        if (!$afiliasi) {
            return response()->json([
                'status' => false,
                'message' => 'Data afiliasi tidak ada!'
            ]);
        }
        
        // cek jamaah
        $jamaah = DB::table('jamaahs')->where('no_ktp', $request->no_ktp)->first();
        if ($jamaah) {
            return response()->json([
                'status' => false,
                'message' => 'Data jamaah sudah ada!'
            ]);
        }

        // cek expired
        if ($request->tanggal_keberangkatan > $request->tanggal_expired) {
            return response()->json([
                'status' => false,
                'message' => 'Passport sudah expired tidak ada!'
            ]);
        }

        DB::beginTransaction();

        try {
            // Cari berdasarkan no_ktp
            $jamaah = Jamaah::where('no_ktp', $request->no_ktp)->first();

            // Buat kode keberangkatan random 5 karakter (huruf dan angka)
            $kodeKeberangkatan = 'ELQ_' . strtoupper(Str::random(3));

            if (!$jamaah) {
                // Insert ke jamaahs
                $jamaah = Jamaah::create([
                    'id_afiliasi'        => $request->id_afiliasi,
                    'title'              => $request->title,
                    'nama'               => $request->nama,
                    'tempat_lahir'       => $request->tempat_lahir,
                    'tanggal_lahir'      => $request->tanggal_lahir,
                    'no_ktp'             => $request->no_ktp,
                    'alamat'             => $request->alamat,
                    'rt'                 => $request->rt,
                    'rw'                 => $request->rw,
                    'kelurahan'          => $request->kelurahan,
                    'kecamatan'          => $request->kecamatan,
                    'kota'               => $request->kota,
                    'provinsi'           => $request->provinsi,
                    'no_passport'        => $request->no_passport,
                    'tanggal_iss'        => $request->tanggal_iss,
                    'tanggal_expired'    => $request->tanggal_expired,
                    'tempat_iss'         => $request->tempat_iss,
                    'status_perkawinan'  => $request->status_perkawinan,
                    'pekerjaan'          => $request->pekerjaan,
                    'nama_ayah'          => $request->nama_ayah,
                ]);
            }

            // Insert ke keberangkatans
            Keberangkatan::create([
                'jenis_umroh'           => $request->jenis_umroh,
                'no_ktp'                => $request->no_ktp,
                'tanggal_keberangkatan' => $request->tanggal_keberangkatan,
                'status_pembayaran'     => $request->status_pembayaran ?? 'Belum Lunas',
                'tanggal_pembayaran'    => $request->tanggal_pembayaran,
                'id_afiliasi'           => $request->id_afiliasi,
                'kode_keberangkatan'    => $kodeKeberangkatan,
            ]);
            
            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Data berhasil disimpan',
                'kode_keberangkatan' => $kodeKeberangkatan
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
        
    }

    /**
     * Display the specified resource.
     */
    public function show($jamaah)
    {
        $data = DB::table('keberangkatans')
            ->select(
                'jamaahs.title', 'jamaahs.nama', 'jamaahs.no_passport', 'jamaahs.tempat_lahir',
                'jamaahs.tanggal_lahir', 'jamaahs.tanggal_iss', 'jamaahs.tanggal_expired',
                'jamaahs.tempat_iss', 'jamaahs.no_ktp', 'jamaahs.alamat', 'jamaahs.rt',
                'jamaahs.rw', 'jamaahs.kelurahan', 'jamaahs.kecamatan', 'jamaahs.kota',
                'jamaahs.provinsi', 'jamaahs.status_perkawinan', 'jamaahs.pekerjaan', 'jamaahs.nama_ayah'
            )
            ->join('jamaahs', 'keberangkatans.no_ktp', '=', 'jamaahs.no_ktp')
            ->where('jamaahs.no_ktp', $jamaah)
            ->first();

        if (!$data) {
            return redirect()->route('jamaah.index')->with('error', 'Data tidak ditemukan!');
        }

        return view('jamaah.show', compact('data'));
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit($no_ktp)
    {
        $jamaah = Jamaah::where('no_ktp', $no_ktp)->firstOrFail();
        return view('jamaah.edit', compact('jamaah'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $no_ktp)
    {
        // Cari data berdasarkan no_ktp
        $afiliasi = DB::table('afiliasis')->where('kode_afiliasi', $request->id_afiliasi)->first();
        if (!$afiliasi) {
            return response()->json([
                'status' => false,
                'message' => 'Data afiliasi tidak ada!'
            ]);
        }
        
        // cek jamaah
        $dataJamaah = DB::table('jamaahs')->where('no_ktp', $no_ktp)->first();
        if (!$dataJamaah) {
            return response()->json([
                'status'  => false,
                'message' => 'Data jamaah tidak ditemukan!',
            ], 404);
        }

        $exists = Jamaah::where('no_ktp', $request->no_ktp)
                    ->where('no_ktp', '!=', $no_ktp)
                    ->exists();
            if ($exists) {
                return response()->json([
                    'status'  => false,
                    'message' => 'No KTP baru sudah terdaftar!',
                ], 422);
            }

        // cek expired
        if ($request->tanggal_keberangkatan > $request->tanggal_expired) {
            return response()->json([
                'status' => false,
                'message' => 'Passport sudah expired tidak ada!'
            ]);
        }

         try {
        DB::beginTransaction();

        // 4. Update data jamaah berdasarkan ID
        $updated = DB::table('jamaahs')
                   ->where('id', $dataJamaah->id) // Update berdasarkan ID
                   ->update([
                       'id_afiliasi'        => $afiliasi->id,
                       'title'              => $request->title,
                       'nama'               => $request->nama,
                       'tempat_lahir'       => $request->tempat_lahir,
                       'tanggal_lahir'      => $request->tanggal_lahir,
                       'no_ktp'             => $request->no_ktp,
                       'alamat'             => $request->alamat,
                       'rt'                 => $request->rt,
                       'rw'                 => $request->rw,
                       'kelurahan'          => $request->kelurahan,
                       'kecamatan'          => $request->kecamatan,
                       'kota'               => $request->kota,
                       'provinsi'           => $request->provinsi,
                       'no_passport'        => $request->no_passport,
                       'tanggal_iss'        => $request->tanggal_iss,
                       'tanggal_expired'    => $request->tanggal_expired,
                       'tempat_iss'         => $request->tempat_iss,
                       'status_perkawinan'  => $request->status_perkawinan,
                       'pekerjaan'          => $request->pekerjaan,
                       'nama_ayah'          => $request->nama_ayah,
                   ]);

        if (!$updated) {
            return response()->json([
                'status'  => false,
                'message' => 'gagal update jamaah.',
            ], 404);
        }

        // 5. Update data keberangkatan jika no_ktp berubah
        if ($no_ktp != $request->no_ktp) {
            DB::table('keberangkatans')
              ->where('no_ktp', $no_ktp)
              ->update(['no_ktp' => $request->no_ktp]);
        }

        DB::commit();

        return response()->json([
            'status'  => true,
            'message' => 'Data jamaah dan keberangkatan berhasil diperbarui.',
        ], 200);

        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'status'  => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage(),
            ], 500);
        }
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $jamaah = Jamaah::findOrFail($id);
        $jamaah->delete();

        return redirect()->route('jamaah.index')->with('success', 'Jamaah berhasil dihapus');
    }

    public function search(Request $request)
    {
        $search = $request->search;
        $data = Jamaah::where('no_ktp', 'like', "%$search%")
                                    ->orWhere('nama', 'like', "%$search%")
                                    ->orWhere('no_passport', 'like', "%$search%")
                                    ->orWhere('kota', 'like', "%$search%")
                                    ->paginate(5);
        
        return view('jamaah.index', compact('data'));
    }

    public function cek(Request $request)
    {
        // Mengambil nilai dari form dengan nama 'kode_keberangkatan'
        $kode_keberangkatan = $request->input('kode_keberangkatan');

        // Melakukan join antara tabel keberangkatans dan jamaahs berdasarkan no_ktp
        $data = DB::table('keberangkatans')
            ->join('jamaahs', 'keberangkatans.no_ktp', '=', 'jamaahs.no_ktp')
            ->where('keberangkatans.kode_keberangkatan', $kode_keberangkatan)
            ->select(
                'keberangkatans.kode_keberangkatan',
                'jamaahs.nama',
                'keberangkatans.tanggal_keberangkatan',
                'keberangkatans.status_pembayaran'
            )  // Memilih kolom yang spesifik
            ->first();

        // Jika tidak ditemukan, kembalikan status false
        if (!$data) {
            return response()->json([
                'status' => false,
                'message' => 'Kode keberangkatan tidak ditemukan.'
            ]);
        }

        return response()->json([
            'status' => true,
            'result' => $data
        ]);
    }

    public function export(Request $request)
    {
        // Validasi input
        $validated = $request->validate([
            'start_tanggal' => 'nullable|date',
            'end_tanggal' => 'nullable|date|after_or_equal:start_tanggal',
            'kota' => 'nullable|string|max:255',
        ]);

        // Proses export
        return Excel::download(
            new JamaahExport($validated['start_tanggal'] ?? null, $validated['end_tanggal'] ?? null, $validated['kota'] ?? null),
            'data-jamaah.xlsx'
        );
    }


}
