<?php

namespace App\Http\Controllers;

use App\Models\Afiliasi;
use App\Http\Requests\StoreAfiliasiRequest;
use App\Http\Requests\UpdateAfiliasiRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\AfiliasiExport;
use Illuminate\Support\Str;

class AfiliasiController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // CTE untuk menghitung jumlah keberangkatan per afiliasi
        $afiliasiWithCount = DB::table('keberangkatans')
            ->select('id_afiliasi', DB::raw('COUNT(*) as jumlah_keberangkatan'))
            ->groupBy('id_afiliasi');

        // Query utama afiliasi
        $query = DB::table('afiliasis as a')
            ->leftJoinSub($afiliasiWithCount, 'k', function ($join) {
                $join->on('a.id', '=', 'k.id_afiliasi');
            })
            ->select('a.*', DB::raw('COALESCE(k.jumlah_keberangkatan, 0) as jumlah_keberangkatan'))
            ->orderByDesc('a.created_at');

        // Filter by kota (opsional)
        if ($request->filled('kota')) {
            $query->where('a.kota', $request->kota);
        }

        // Paginate hasilnya
        $data = $query->paginate(5);

        return view('afiliasi.index', compact('data'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('afiliasi.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $kodeAfiliasi = 'AF_' . strtoupper(Str::random(3));
        
        Afiliasi::create([
            'nama' => $request->nama,
            'email' => $request->email,
            'tanggal_lahir' => $request->tanggal_lahir,
            'alamat' => $request->alamat,
            'no_ktp' => $request->no_ktp,
            'no_telepon' => $request->no_telepon,
            'kota' => $request->kota,
            'kode_afiliasi' => $kodeAfiliasi
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Data afiliasi berhasil disimpan!',
            'kode_afiliasi' => $kodeAfiliasi
        ]);
    }


    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $afiliasi = Afiliasi::findOrFail($id);
        return view('afiliasi.show', compact('afiliasi'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $afiliasi = Afiliasi::where('id', $id)->firstOrFail();
        return view('afiliasi.edit', compact('afiliasi'));
    }


    public function update(Request $request, $id)
    {
        
        $afiliasi = Afiliasi::findOrFail($id);
        try {
            
            $afiliasi->update([
                'nama' => $request->nama,
                'email' => $request->email,
                'tanggal_lahir' => $request->tanggal_lahir,
                'alamat' => $request->alamat,
                'no_ktp' => $request->no_ktp,
                'no_telepon' => $request->no_telepon,
                'kota' => $request->kota
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Data afiliasi berhasil diperbarui!',
                'data' => $afiliasi
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal memperbarui data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $afiliasi = Afiliasi::findOrFail($id);
        $afiliasi->delete();

        return redirect()->route('afiliat.index')->with('success', 'Data berhasil dihapus');
    }

    public function search(Request $request)
    {
        $search = $request->search;
        $data = Afiliasi::where('no_ktp', 'like', "%$search%")
                                    ->orWhere('nama', 'like', "%$search%")
                                    ->orWhere('kode_afiliasi', 'like', "%$search%")
                                    ->orWhere('kota', 'like', "%$search%")
                                    ->paginate(5);
        
        return view('afiliasi.index', compact('data'));
    }

    public function export(Request $request)
    {
        $validated = $request->validate([
            'start_tanggal' => 'nullable|date',
            'end_tanggal' => 'nullable|date|after_or_equal:start_tanggal',
            'kota' => 'nullable|string|max:255',
        ]);

        return Excel::download(
            new AfiliasiExport(
                $validated['start_tanggal'] ?? null, 
                $validated['end_tanggal'] ?? null, 
                $validated['kota'] ?? null
            ),
            'data-Afiliasi.xlsx'
        );
    }

}
